<?php

namespace App\Http\Controllers;

use App\Models\CourseCategory;
use Illuminate\Http\Request;
use Illuminate\Http\JsonResponse;
use Illuminate\Support\Facades\Validator;

class CourseCategoryController extends Controller
{
    public function index(): JsonResponse
    {
        $categories = CourseCategory::with('children')->get()->map(function ($category) {
            // Decode Metadata to return as a clean JSON object
            if ($category->Metadata && is_string($category->Metadata)) {
                $category->Metadata = json_decode($category->Metadata, true);
            }
            return $category;
        });

        return response()->json([
            'status' => 'success',
            'data' => $categories,
        ], 200);
    }

    public function show($id): JsonResponse
    {
        $category = CourseCategory::with('children')->findOrFail($id);

        // Decode Metadata to return as a clean JSON object
        if ($category->Metadata && is_string($category->Metadata)) {
            $category->Metadata = json_decode($category->Metadata, true);
        }

        return response()->json([
            'status' => 'success',
            'data' => $category,
        ], 200);
    }

    public function store(Request $request): JsonResponse
    {
        $validator = Validator::make($request->all(), [
            'Name' => 'required|string|max:255',
            'Description' => 'sometimes|string|nullable',
            'Parent_id' => 'sometimes|nullable|exists:course_categories,Category_id',
            'Metadata' => 'sometimes|nullable',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'Name',
            'Description',
            'Parent_id',
            'Metadata',
        ]);

        // Convert Metadata to JSON string if it's an array or object
        if (isset($data['Metadata']) && (is_array($data['Metadata']) || is_object($data['Metadata']))) {
            $data['Metadata'] = json_encode($data['Metadata'], JSON_UNESCAPED_SLASHES);
        } elseif (isset($data['Metadata']) && is_string($data['Metadata'])) {
            // Verify if string is valid JSON, otherwise set to null
            json_decode($data['Metadata']);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $data['Metadata'] = null;
            }
        }

        $category = CourseCategory::create($data);

        // Decode Metadata for response
        if ($category->Metadata && is_string($category->Metadata)) {
            $category->Metadata = json_decode($category->Metadata, true);
        }

        return response()->json([
            'status' => 'success',
            'data' => $category,
        ], 201);
    }

    public function update(Request $request, $id): JsonResponse
    {
        $category = CourseCategory::findOrFail($id);

        $validator = Validator::make($request->all(), [
            'Name' => 'sometimes|string|max:255',
            'Description' => 'sometimes|string|nullable',
            'Parent_id' => 'sometimes|nullable|exists:course_categories,Category_id',
            'Metadata' => 'sometimes|nullable',
        ]);

        if ($validator->fails()) {
            return response()->json([
                'status' => 'error',
                'errors' => $validator->errors(),
            ], 422);
        }

        $data = $request->only([
            'Name',
            'Description',
            'Parent_id',
            'Metadata',
        ]);

        // Convert Metadata to JSON string if it's an array or object
        if (isset($data['Metadata']) && (is_array($data['Metadata']) || is_object($data['Metadata']))) {
            $data['Metadata'] = json_encode($data['Metadata'], JSON_UNESCAPED_SLASHES);
        } elseif (isset($data['Metadata']) && is_string($data['Metadata'])) {
            // Verify if string is valid JSON, otherwise set to null
            json_decode($data['Metadata']);
            if (json_last_error() !== JSON_ERROR_NONE) {
                $data['Metadata'] = null;
            }
        }

        $category->update($data);

        // Decode Metadata for response
        if ($category->Metadata && is_string($category->Metadata)) {
            $category->Metadata = json_decode($category->Metadata, true);
        }

        return response()->json([
            'status' => 'success',
            'data' => $category,
        ], 200);
    }

    public function destroy($id): JsonResponse
    {
        $category = CourseCategory::findOrFail($id);
        $category->delete();

        return response()->json([
            'status' => 'success',
            'message' => 'Category deleted successfully',
        ], 200);
    }
}
